﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Media;
using System.Windows.Forms;
using Psion.RFID.HF;

namespace HF_Demo
{
    public sealed partial class ReadIDForm : Form
    {
        //List of (tag UID, count) couples; each element of the ArrayList is an ArrayList

        private readonly SoundPlayer _beepInventory;
        private readonly SoundPlayer _beepNewTag;

        private readonly Reader _reader; //RFID Reader instance
        private readonly ArrayList _listTags;

        private float _avgRdSpeed;
        private bool _btRdStatus; //(Read /Stop reading )


        private int _column;
        private int _cptSound;
        private List<ListViewItem> _lstvwItem;

        private StreamWriter _rdFileStream; //Used to retrive all the tag description xml files from "Tags" folder
        private SoundFrequency _sndFreq;
        private bool _sortDirectionAsc; // Sort direction(ascending descending)
        private int _totalNbtagRead; //Number of tag read in one inventory cycle

        public ReadIDForm(Reader reader)
        {
            InitializeComponent();

            if (!Device.IsWM) mainMenu1.Dispose();

            Height = Screen.PrimaryScreen.WorkingArea.Height;

            // init reader comunication
            _reader = reader;
            _btRdStatus = true;
            _totalNbtagRead = 0;
            _avgRdSpeed = 0;
            _lstvwItem = new List<ListViewItem>();
            _sortDirectionAsc = true; //Sort Direction for "count" column in listview LstVTagRead
            _listTags = new ArrayList();

            //Sound Variables
            _sndFreq = SoundFrequency.LowFreq;
            _cptSound = 0;
            _beepInventory =
                new SoundPlayer(IOHelpers.ApplicationPath + "Sounds/Beep_Inventory.wav");
            _beepNewTag = new SoundPlayer(IOHelpers.ApplicationPath + "Sounds/Beep_NewTag.wav");
        }

        #region Non-RFID Enhancements

        /// <summary>
        /// Updates information in listview LstVTagRead
        /// </summary>
        private void Displayer()
        {
            int cptChips = 0, cptChipsDiff = 0;

            foreach (ArrayList listTemp in _listTags)
            {
                var lvi = new ListViewItem(new[] {listTemp[0].ToString(), listTemp[1].ToString()});

                if (LstVTagRead.Items.Count == 0)
                    LstVTagRead.Items.Add(lvi);
                else
                {
                    bool boo = false;

                    int cpt;
                    for (cpt = 0; cpt < LstVTagRead.Items.Count; cpt++)
                    {
                        //Check if tag is already in list
                        if (LstVTagRead.Items[cpt].SubItems[0].Text == listTemp[0].ToString())
                        {
                            boo = true;
                            if (LstVTagRead.Items[cpt].SubItems[1].Text != listTemp[1].ToString())
                                LstVTagRead.Items[cpt].SubItems[1].Text = listTemp[1].ToString();
                            else
                                break;
                        }
                    }
                    //If not in list add
                    if (!boo)
                        LstVTagRead.Items.Add(lvi);
                }

                cptChips += (int) listTemp[1];
                cptChipsDiff += 1;
            }

            lbl_SumTagRead.Text = cptChips.ToString(CultureInfo.InvariantCulture);
            lbl_SumDistTagRd.Text = cptChipsDiff.ToString(CultureInfo.InvariantCulture);
            Refresh();
        }

        /// <summary>
        /// Compare the count attribute of 2 items in the 
        /// Listview
        /// </summary>
        /// <param name="item1"></param>
        /// <param name="item2"></param>
        /// <returns>
        /// A signed integer that indicates the relative values of x and y
        /// positiv num if item1 superior item2, negativ if item1 inferior item2...
        /// </returns>
        private int CompareSubItmes(ListViewItem item1,
                                    ListViewItem item2)
        {
            try
            {
                if (_sortDirectionAsc)
                    return
                        (
                            Convert.ToInt32(item1.SubItems[1].Text) - Convert.ToInt32(item2.SubItems[1].Text)
                        );

                return
                    (
                        -Convert.ToInt32(item1.SubItems[1].Text) + Convert.ToInt32(item2.SubItems[1].Text)
                    );
            }
            catch (Exception ex)
            {
                lbl_TagFound.Text = "";
                //Show Exception message Once
                lbl_TagFound.ForeColor = Color.Red;
                lbl_TagFound.Text = ex.Message;

                return -1;
            }
        }

        /// <summary>
        /// Compare the tagID attribute of 2 items in the 
        /// Listview
        /// </summary>
        /// <param name="item1"></param>
        /// <param name="item2"></param>
        /// <returns>
        /// A signed integer that indicates the relative values of x and y
        /// positiv num if item1 superior item2, negativ if item1 inferior item2...
        /// </returns>
        private int CompareSubItmes2(ListViewItem item1,
                                     ListViewItem item2)
        {
            try
            {
                if (_sortDirectionAsc)
                    return string.Compare(item1.SubItems[0].Text, item2.SubItems[0].Text);

                return (-string.Compare(item1.SubItems[0].Text, item2.SubItems[0].Text));
            }
            catch (Exception ex)
            {
                lbl_TagFound.Text = "";

                //Show Exception message Once
                lbl_TagFound.ForeColor = Color.Red;
                lbl_TagFound.Text = ex.Message;

                return -1;
            }
        }

        /// <summary>
        /// Sort the listview content
        /// </summary>
        private void Sort()
        {
            //CHECK IF SUBITEM NOT EMPTY
            if (_lstvwItem.Count > 1)
            {
                //Temporary list for sorting purposes
                //templist:put only tagIDs not error messages(which have only 1 subitem)
                List<ListViewItem> tempList = _lstvwItem.Where(item => item.SubItems.Count > 1).ToList();


                //Copy the filtered list back to lstvwItem
                _lstvwItem = tempList;

                if (_lstvwItem.Count > 1)
                {
                    if (_column == 0)
                        _lstvwItem.Sort(CompareSubItmes2);
                    else if (_column == 1)
                        _lstvwItem.Sort(CompareSubItmes);
                }
            }
        }

        /// <summary>
        ///  Color the Listview content
        /// </summary>
        /// <param name="toWhite"></param>
        private void ColorListView(bool toWhite)
        {
            bool bckColorAltern = false;

            for (int i = 0; i < LstVTagRead.Items.Count && LstVTagRead.Items.Count > 1; i++)
            {
                if (!toWhite)
                {
                    if (bckColorAltern)
                    {
                        LstVTagRead.Items[i].BackColor = Color.SkyBlue;
                        bckColorAltern = false;
                    }
                    else
                    {
                        LstVTagRead.Items[i].BackColor = Color.PeachPuff;
                        bckColorAltern = true;
                    }
                }
                else
                {
                    LstVTagRead.Items[i].BackColor = Color.White;
                }
            }
        }

        #endregion

        #region Events

        /// <summary>
        /// Handles the Click event of the bt_read control.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        private void BtReadClick(object sender, EventArgs e)
        {
            Cursor.Current = Cursors.WaitCursor;
            //Read Mode
            if (_btRdStatus)
            {
                InventoryTimer.Enabled = true;
                //soundTimer.Enabled = true;
                bt_read.Text = "STOP";
                _btRdStatus = false;
            }

                //Stop-Read Mode
            else
            {
                bt_read.Text = "READ";
                _btRdStatus = true;
                InventoryTimer.Enabled = false;
                soundTimer.Enabled = false;
                lbl_TagFound.Text = "";
            }
            Cursor.Current = Cursors.Default;
        }

        /// <summary>
        /// Handles the Click event of the bt_clr control.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        private void BtClrClick(object sender, EventArgs e)
        {
            LstVTagRead.Items.Clear();
            _listTags.Clear();
            lbl_rdSpd.Text = "";
            lbl_SumDistTagRd.Text = "0";
            lbl_SumTagRead.Text = "0";
            lbl_TagFound.Text = "";
        }

        /// <summary>
        /// Handles the Click event of the bt_save control.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        private void BtSaveClick(object sender, EventArgs e)
        {
            if (saveFileDialog1.ShowDialog() == DialogResult.OK)
            {
                //retrieve file name and file directory
                //save the data in the listview
                try
                {
                    if (saveFileDialog1.FilterIndex == 1)
                        _rdFileStream = new StreamWriter(saveFileDialog1.FileName + ".csv", true);
                    else if (saveFileDialog1.FilterIndex == 2)
                        _rdFileStream = new StreamWriter(saveFileDialog1.FileName + ".txt", true);

                    _rdFileStream.Flush();
                    _rdFileStream.Write(DateTime.Now);
                    _rdFileStream.WriteLine();
                    _rdFileStream.Write("Number of distinct tag read:" + lbl_SumDistTagRd.Text);
                    _rdFileStream.WriteLine();
                    _rdFileStream.Write("Number of tag read:" + lbl_SumTagRead.Text);
                    _rdFileStream.WriteLine();
                    _rdFileStream.WriteLine();
                    _rdFileStream.Write("TAG ID READ,COUNT");
                    _rdFileStream.WriteLine();
                    foreach (ListViewItem obj in LstVTagRead.Items)
                    {
                        _rdFileStream.Write(obj.Text + "," + obj.SubItems[1].Text);
                        _rdFileStream.WriteLine();
                    }
                    _rdFileStream.Flush();
                    _rdFileStream.Close();
                }
                catch (Exception ex)
                {
                    MessageBox.Show(ex.Message);
                }
            }
        }

        /// <summary>
        /// Handles the Tick event of the InventoryTimer control.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        private void InventoryTimerTick(object sender, EventArgs e)
        {
            int t = Environment.TickCount;

            ReadTagID();

            TimeSpan w = TimeSpan.FromMilliseconds(Environment.TickCount - t);

            _avgRdSpeed = _totalNbtagRead/(w.Seconds + (float) w.Milliseconds/1000);

            if (_totalNbtagRead == 0)
                _sndFreq = SoundFrequency.LowFreq;

            else if (_totalNbtagRead <= 5 && _totalNbtagRead > 0)
                _sndFreq = SoundFrequency.HighFreq;

            else if (_totalNbtagRead > 5)
                _sndFreq = SoundFrequency.VeryHighFreq;

            Displayer();
            _totalNbtagRead = 0;

            //Application.DoEvents();
            PlaySound();
        }

        /// <summary>
        /// Handles the Tick event of the soundTimer control.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        private void SoundTimerTick(object sender, EventArgs e)
        {
            PlaySound();
        }

        private void PlaySound()
        {
            if (_avgRdSpeed > 0)
                lbl_rdSpd.Text = string.Format("Read Speed: {0:.00} tags/s", _avgRdSpeed);
            else lbl_rdSpd.Text = "Read Speed: 0 tags/s";

            switch (_sndFreq)
            {
                case (SoundFrequency.LowFreq):
                    {
                        //if (_cptSound % 14 == 0) _beepRfon.Play();
                    }
                    break;
                case (SoundFrequency.HighFreq):
                    {
                        //if (_cptSound % 3 == 0) _beepInventory.Play();
                        _beepInventory.Play();
                    }
                    break;
                case (SoundFrequency.VeryHighFreq):
                    {
                        _beepInventory.Play();
                    }
                    break;
            }

            _cptSound++;
        }

        /// <summary>
        /// Handles the Closing event of the ReadIDForm control.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">The <see cref="System.ComponentModel.CancelEventArgs"/> instance containing the event data.</param>
        private void ReadIDFormClosing(object sender, CancelEventArgs e)
        {
            //If there's an ongoing read stop it before exiting
            if (!_btRdStatus) BtReadClick(new object(), new EventArgs());
        }

        /// <summary>
        /// Handles the ColumnClick event of the LstVTagRead control.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">The <see cref="System.Windows.Forms.ColumnClickEventArgs"/> instance containing the event data.</param>
        private void LstVTagReadColumnClick(object sender, ColumnClickEventArgs e)
        {
            //Count Column
            _column = e.Column;
            if (e.Column == 1 || e.Column == 0)
            {
                _lstvwItem.Clear();
                foreach (ListViewItem obj in LstVTagRead.Items)
                {
                    _lstvwItem.Add(obj);
                }
                try
                {
                    Sort();
                }
                catch (Exception)
                {
                    lbl_TagFound.ForeColor = Color.Red;
                    lbl_TagFound.Text = "Sorting Error";
                    return;
                }
                LstVTagRead.Items.Clear();
                foreach (ListViewItem t in _lstvwItem)
                {
                    LstVTagRead.Items.Add(t);
                }
                //colorListView(false);
                _sortDirectionAsc = !_sortDirectionAsc;
            }
        }

        /// <summary>
        /// Handles the Click event of the menuItem1 control.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        private void MenuItem1Click(object sender, EventArgs e)
        {
            Close();
        }

        #endregion

        #region RFID

        /// <summary>
        /// Reads the tag ID.
        /// </summary>
        private void ReadTagID()
        {
            lbl_TagFound.Text = "";
            byte prefix;
            string tagID;

            try
            {
                byte[] data = _reader.Protocol("s");

                RFIDHelpers.CheckErrors(data);

                prefix = data[0];
                tagID = BitConverter.ToString(data).Replace("-", "");
                // suppress tag type (prefix)
                tagID = tagID.Substring(2, tagID.Length - 2);
            }
            catch (Exception ex)
            {
                lbl_TagFound.Text = "";

                if (ex.Message.Equals("No tag in the field"))
                {
                    //Show Exception message Once
                    lbl_TagFound.ForeColor = Color.DeepSkyBlue;
                    lbl_TagFound.Text = ex.Message;
                }
                else
                {
                    //Show Exception message Once
                    lbl_TagFound.ForeColor = Color.Red;
                    lbl_TagFound.Text = ex.Message;
                }
                return;
            }

            if (tagID.Length > 2)
            {
                if (checkBoxTagType.Checked)
                {
                    string tagType = RFIDHelpers.GetTagType(Convert.ToByte(prefix));
                    tagType += ": ";

                    tagID = tagType + tagID;
                }

                var it = new ListViewItem();
                bool notInList = true;

                lbl_TagFound.ForeColor = Color.Lime;
                lbl_TagFound.Text = "FOUND";
                lbl_TagFound.Refresh();

                //Sound Timer
                _totalNbtagRead++;

                lbl_SumTagRead.Text = (Convert.ToInt32(lbl_SumTagRead.Text) + 1).ToString();
                it.Text = tagID;

                foreach (ArrayList listTemp in _listTags)
                {
                    if (listTemp[0].ToString() == tagID)
                    {
                        listTemp[1] = (int) listTemp[1] + 1;
                        notInList = false;
                        break;
                    }
                }

                //TagID not in List, it will be added as a new item
                if (notInList)
                {
                    _beepNewTag.Play();
                    var detailTag = new ArrayList {tagID, 1};
                    _listTags.Add(detailTag);
                }
            }
        }


        #endregion
    }

    public enum SoundFrequency
    {
        HighFreq,
        LowFreq,
        VeryHighFreq
    }
}